<?php
session_start();
require_once 'config.php';
check_login();

$user_id = $_SESSION['user_id'] ?? null;
if (!$user_id) {
    header('Location: login.php');
    exit;
}

// Türkçe ay isimleri için locale ayarla (sadece *nix sistemlerde etkili)
// Windows için farklı yöntem gerekebilir
setlocale(LC_TIME, 'tr_TR.UTF-8');

// Filtre değerlerini al (GET ile)
$filterYear = isset($_GET['year']) && $_GET['year'] !== '' ? (int)$_GET['year'] : null;
$filterMonth = isset($_GET['month']) && $_GET['month'] !== '' ? (int)$_GET['month'] : null;

// Dinamik WHERE koşulu oluştur
$where = "h.user_id = :user_id";
$params = ['user_id' => $user_id];

if ($filterYear !== null) {
    $where .= " AND h.year = :year";
    $params['year'] = $filterYear;
}

if ($filterMonth !== null) {
    $where .= " AND h.month = :month";
    $params['month'] = $filterMonth;
}

// Sorgu: alışkanlık başlığı, yıl, ay, işaretlenen sayısı
$sql = "
  SELECT h.title, h.year, h.month, COUNT(hc.checked) AS total_checked
  FROM habits h
  LEFT JOIN habit_checkmarks hc ON h.id = hc.habit_id AND hc.checked = 1
  WHERE $where
  GROUP BY h.title, h.year, h.month
  ORDER BY h.title, h.year DESC, h.month DESC
";
$stmt = $pdo->prepare($sql);
$stmt->execute($params);

// Veriyi hazırla
$data = [];
$labels = [];

while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $title = $row['title'];
    $year = $row['year'];
    $month = $row['month'];
    $key = sprintf('%04d-%02d', $year, $month);

    if (!in_array($key, $labels)) {
        $labels[] = $key;
    }

    if (!isset($data[$title])) {
        $data[$title] = [];
    }

    $data[$title][$key] = (int)$row['total_checked'];
}

// Tarihleri büyükten küçüğe sırala
rsort($labels);

// Eksik tarihleri 0 ile doldur
foreach ($data as $title => &$values) {
    foreach ($labels as $label) {
        if (!isset($values[$label])) {
            $values[$label] = 0;
        }
    }
    // Tarihleri yine büyükten küçüğe sırala
    uksort($values, function($a, $b) {
        return strcmp($b, $a);
    });
}
unset($values);

// Kullanıcı adı ve soyadı header için
$ad = $_SESSION['ad'] ?? '';
$soyad = $_SESSION['soyad'] ?? '';

// Yıl ve ay seçimi için seçenekler
$currentYear = (int)date('Y');
$currentMonth = (int)date('m');

// Aylar Türkçe olarak
$aylar = [
    1 => 'Ocak', 2 => 'Şubat', 3 => 'Mart', 4 => 'Nisan',
    5 => 'Mayıs', 6 => 'Haziran', 7 => 'Temmuz', 8 => 'Ağustos',
    9 => 'Eylül', 10 => 'Ekim', 11 => 'Kasım', 12 => 'Aralık'
];
?>

<!DOCTYPE html>
<html lang="tr">
<head>
<meta charset="UTF-8" />
<title>Alışkanlık İstatistikleri</title>
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<style> body {
    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    background: #f7f9fb;
    margin: 0; padding: 0;
  }
  header {
    background: #4a90e2;
    color: white;
    padding: 15px 30px;
    display: flex;
    justify-content: space-between;
    align-items: center;
  }
  header h1 {
    margin: 0;
    font-weight: normal;
  }
  header .user-info {
    font-size: 1.1em;
  }
  main {
    max-width: 960px;
    margin: 40px auto;
    background: white;
    border-radius: 12px;
    padding: 30px;
    box-shadow: 0 0 10px #ccc;
  }
  .filters {
    margin-bottom: 25px;
    display: flex;
    gap: 15px;
    align-items: center;
  }
  .filters label {
    font-weight: 600;
  }
  select {
    padding: 6px 10px;
    border-radius: 6px;
    border: 1.5px solid #ccc;
    font-size: 1em;
  }
  button {
    background: #4a90e2;
    color: white;
    border: none;
    padding: 8px 15px;
    border-radius: 6px;
    cursor: pointer;
    font-weight: 600;
    transition: background-color 0.3s ease;
  }
  button:hover {
    background: #357abd;
  }
  .chart-container {
    width: 100%;
  }
</style>
</head>
<body>

<header>
  <h1>Alışkanlık İstatistikleri</h1>
  <div class="user-info"><?= htmlspecialchars($ad . ' ' . $soyad) ?></div>
</header>

<main>

  <form method="GET" class="filters" id="filterForm">
    <label for="year">Yıl:</label>
    <select name="year" id="year">
      <option value="">Tümü</option>
      <?php for ($y = $currentYear; $y >= 2000; $y--): ?>
        <option value="<?= $y ?>" <?= ($filterYear === $y) ? 'selected' : '' ?>><?= $y ?></option>
      <?php endfor; ?>
    </select>

    <label for="month">Ay:</label>
    <select name="month" id="month">
      <option value="">Tümü</option>
      <?php foreach ($aylar as $num => $isim): ?>
        <option value="<?= $num ?>" <?= ($filterMonth === $num) ? 'selected' : '' ?>><?= $isim ?></option>
      <?php endforeach; ?>
    </select>

    <button type="submit">Filtrele</button>

  </form>

  <?php if (empty($data)): ?>
    <p>Gösterilecek veri yok.</p>
  <?php else: ?>
    <div class="chart-container">
      <canvas id="habitChart"></canvas>
    </div>
  <?php endif; ?>

  <div style="margin-top: 20px;">
    <button onclick="window.location.href='index.php'">Anasayfaya Dön</button>
  </div>

</main>

<script>
const ctx = document.getElementById('habitChart').getContext('2d');

const labels = <?= json_encode($labels) ?>;

// Etiketleri 'YYYY-MM' den 'MM.YYYY' veya 'Ay Yıl' formatına çevirmek için:
const turkishMonths = <?= json_encode($aylar) ?>;

const formattedLabels = labels.map(l => {
  const parts = l.split('-');
  const y = parts[0];
  const m = parseInt(parts[1], 10);
  return turkishMonths[m] + ' ' + y;
});

const datasets = [
  <?php foreach ($data as $title => $values): ?>
  {
    label: <?= json_encode($title) ?>,
    data: <?= json_encode(array_values($values)) ?>,
    fill: false,
    borderColor: 'hsl(<?= rand(0, 360) ?>, 70%, 50%)',
    tension: 0.3,
    pointRadius: 3,
    pointHoverRadius: 6,
  },
  <?php endforeach; ?>
];

new Chart(ctx, {
  type: 'line',
  data: {
    labels: formattedLabels,
    datasets: datasets
  },
  options: {
    responsive: true,
    plugins: {
      title: {
        display: true,
        text: 'Alışkanlık Takip Grafiği',
        font: { size: 20 }
      },
      legend: {
        position: 'bottom'
      },
      tooltip: {
        mode: 'nearest',
        intersect: false,
      }
    },
    interaction: {
      mode: 'nearest',
      axis: 'x',
      intersect: false
    },
    scales: {
      y: {
        beginAtZero: true,
        title: {
          display: true,
          text: 'İşaretleme Sayısı'
        }
      },
      x: {
        title: {
          display: true,
          text: 'Ay - Yıl'
        },
        ticks: {
          maxRotation: 90,
          minRotation: 45
        }
      }
    }
  }
});
</script>

</body>
</html>
